// SPI-COPOSITE demonstration program for PIC32MK1024GPD064 by K.Tanaka
#include <xc.h>
#include <sys/attribs.h>
#include "spi_composite.h"

// DEVCFG3
// USERID = No Setting
#pragma config FUSBIDIO2 = OFF          // USB2 USBID Selection (USBID pin is controlled by the port function)
#pragma config FVBUSIO2 = OFF           // USB2 VBUSON Selection bit (VBUSON pin is controlled by the port function)
#pragma config PGL1WAY = OFF            // Permission Group Lock One Way Configuration bit (Allow multiple reconfigurations)
#pragma config PMDL1WAY = OFF           // Peripheral Module Disable Configuration (Allow multiple reconfigurations)
#pragma config IOL1WAY = OFF            // Peripheral Pin Select Configuration (Allow multiple reconfigurations)
#pragma config FUSBIDIO1 = OFF          // USB1 USBID Selection (USBID pin is controlled by the port function)
#pragma config FVBUSIO1 = OFF           // USB2 VBUSON Selection bit (VBUSON pin is controlled by the port function)

// DEVCFG2
#pragma config FPLLIDIV = DIV_1         // System PLL Input Divider (1x Divider)
#pragma config FPLLRNG = RANGE_8_16_MHZ // System PLL Input Range (8-16 MHz Input)
#pragma config FPLLICLK = PLL_POSC      // System PLL Input Clock Selection (POSC is input to the System PLL)
#pragma config FPLLMULT = MUL_32        // System PLL Multiplier (PLL Multiply by 32)
#pragma config FPLLODIV = DIV_4         // System PLL Output Clock Divider (4x Divider)
#pragma config VBATBOREN = OFF          // VBAT BOR Enable (Disable ZPBOR during VBAT Mode)
#pragma config DSBOREN = OFF            // Deep Sleep BOR Enable (Disable ZPBOR during Deep Sleep Mode)
#pragma config DSWDTPS = DSPS32         // Deep Sleep Watchdog Timer Postscaler (1:2^36)
#pragma config DSWDTOSC = LPRC          // Deep Sleep WDT Reference Clock Selection (Select LPRC as DSWDT Reference clock)
#pragma config DSWDTEN = OFF            // Deep Sleep Watchdog Timer Enable (Disable DSWDT during Deep Sleep Mode)
#pragma config FDSEN = OFF              // Deep Sleep Enable (Disable DSEN bit in DSCON)
#pragma config UPLLEN = OFF             // USB PLL Enable (USB PLL Disabled)

// DEVCFG1
#pragma config FNOSC = SPLL             // Oscillator Selection Bits (System PLL)
#pragma config DMTINTV = WIN_127_128    // DMT Count Window Interval (Window/Interval value is 127/128 counter value)
#pragma config FSOSCEN = OFF            // Secondary Oscillator Enable (Disable Secondary Oscillator)
#pragma config IESO = ON                // Internal/External Switch Over (Enabled)
#pragma config POSCMOD = HS             // Primary Oscillator Configuration (HS osc mode)
#pragma config OSCIOFNC = OFF           // CLKO Output Signal Active on the OSCO Pin (Disabled)
#pragma config FCKSM = CSECME           // Clock Switching and Monitor Selection (Clock Switch Enabled, FSCM Enabled)
#pragma config WDTPS = PS1048576        // Watchdog Timer Postscaler (1:1048576)
#pragma config WDTSPGM = STOP           // Watchdog Timer Stop During Flash Programming (WDT stops during Flash programming)
#pragma config WINDIS = NORMAL          // Watchdog Timer Window Mode (Watchdog Timer is in non-Window mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (WDT Disabled)
#pragma config FWDTWINSZ = WINSZ_25     // Watchdog Timer Window Size (Window size is 25%)
#pragma config DMTCNT = DMT31           // Deadman Timer Count Selection (2^31 (2147483648))
#pragma config FDMTEN = OFF             // Deadman Timer Enable (Deadman Timer is disabled)

// DEVCFG0
#pragma config DEBUG = OFF              // Background Debugger Enable (Debugger is disabled)
#pragma config JTAGEN = OFF             // JTAG Enable (JTAG Disabled)
#pragma config ICESEL = ICS_PGx1        // ICE/ICD Comm Channel Select (Communicate on PGEC1/PGED1)
#pragma config TRCEN = OFF              // Trace Enable (Trace features in the CPU are disabled)
#pragma config BOOTISA = MIPS32         // Boot ISA Selection (Boot code and Exception code is MIPS32)
#pragma config FSLEEP = OFF             // Flash Sleep Mode (Flash is powered down when the device is in Sleep mode)
#pragma config SMCLR = MCLR_NORM        // Soft Master Clear Enable (MCLR pin generates a normal system Reset)
#pragma config SOSCGAIN = GAIN_2X       // Secondary Oscillator Gain Control bits (2x gain setting)
#pragma config SOSCBOOST = ON           // Secondary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config POSCGAIN = GAIN_LEVEL_0  // Primary Oscillator Gain Control bits (Gain Level 0 (Lowest))
#pragma config POSCBOOST = ON           // Primary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config EJTAGBEN = NORMAL        // EJTAG Boot Enable (Normal EJTAG functionality)

// DEVCP
#pragma config CP = OFF                 // Code Protect (Protection Disabled)

// 32x32ドットキャラクターデータ
const unsigned short bmp1[32*32]=
{
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,5,7,0,5,7,7,5,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,1,6,3,5,7,3,5,2,5,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,5,7,0,7,5,7,6,1,6,5,3,6,1,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,2,5,7,0,7,0,5,7,7,5,7,1,6,1,4,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,5,7,5,7,5,7,7,0,3,0,4,7,7,5,3,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,3,4,3,7,0,7,1,7,5,6,5,7,3,5,0,7,5,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,4,7,5,0,7,7,7,7,0,7,3,5,6,5,7,1,2,7,7,7,
	7,7,7,7,7,7,7,7,7,7,3,5,7,1,6,1,5,7,0,5,7,5,7,0,7,3,4,7,5,7,7,7,
	7,7,7,7,7,7,7,0,3,0,7,2,1,6,3,6,7,5,7,2,5,2,5,7,7,5,7,0,7,7,7,7,
	7,7,7,7,7,7,7,3,6,7,0,7,7,1,3,5,3,0,7,5,7,7,1,7,0,7,1,7,7,0,7,7,
	7,7,7,7,2,0,2,5,3,5,3,6,1,6,0,6,3,4,3,4,3,5,7,0,7,5,7,6,5,7,7,7,
	7,7,7,7,1,7,5,2,6,0,7,1,6,3,7,1,6,3,5,3,7,6,7,5,7,7,1,0,4,3,7,7,
	7,7,5,2,7,2,3,5,3,7,2,7,2,1,2,7,1,6,3,4,5,1,4,3,7,0,4,4,0,4,7,7,
	7,7,7,7,0,3,5,2,7,0,7,0,5,7,0,7,0,3,4,3,2,0,4,0,4,0,4,1,4,7,7,7,
	7,7,2,0,7,7,0,3,0,7,3,1,7,2,7,3,6,1,7,0,4,4,0,4,0,4,5,0,7,7,7,7,
	7,7,5,3,0,7,2,5,2,1,6,2,1,7,0,7,1,6,3,5,3,0,5,0,5,0,4,0,7,7,7,7,
	7,7,7,0,7,0,7,2,1,6,1,7,4,2,1,6,3,0,7,2,7,7,4,0,4,0,7,7,7,7,7,7,
	7,7,7,7,2,3,7,1,6,3,1,6,3,1,7,2,5,2,7,1,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,0,5,2,7,1,6,2,7,7,2,5,3,7,0,0,4,7,7,7,7,7,7,7,7,7,7,7,7,
	7,0,4,0,4,0,4,0,3,5,3,0,3,4,3,6,3,5,0,0,7,7,7,7,0,4,0,5,2,7,7,7,
	1,4,0,4,0,5,0,4,0,2,4,1,4,0,0,1,4,0,4,0,7,7,5,0,4,0,4,0,4,0,7,7,
	7,0,4,1,4,0,4,0,4,7,1,6,0,4,4,4,0,4,4,0,6,7,0,4,0,4,1,4,0,4,0,4,
	7,4,1,4,0,4,1,4,0,0,4,1,4,0,5,0,4,1,0,7,5,0,4,0,4,1,4,0,5,4,0,7,
	7,7,4,0,4,4,4,0,4,3,0,7,0,4,0,4,0,4,7,7,2,5,0,4,0,4,4,4,0,4,1,7,
	7,7,7,4,1,0,4,0,5,7,4,7,7,0,5,0,7,7,7,5,0,4,0,5,0,5,0,0,4,0,7,7,
	7,7,7,7,7,4,1,7,7,7,0,7,7,7,7,7,7,7,7,2,7,7,4,0,4,0,4,5,0,7,7,7,
	7,7,7,7,7,7,7,7,7,7,0,7,7,7,7,7,7,7,0,5,7,7,7,5,0,4,0,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,0,7,7,7,7,7,7,7,4,6,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,0,7,7,7,7,7,7,4,0,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
};

const unsigned short bmp2[32*32]={
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,2,3,2,3,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,3,2,3,0,3,1,2,3,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,3,2,3,0,3,2,2,3,2,3,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,3,7,3,0,3,7,3,0,3,7,7,7,7,7,6,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,3,1,3,2,7,7,7,7,2,3,2,3,7,4,4,4,4,4,6,7,7,7,7,7,
	7,7,7,7,7,7,7,7,2,3,2,0,7,7,7,3,1,0,3,0,4,4,4,6,4,4,4,7,7,7,7,7,
	7,7,7,7,7,7,7,7,3,0,3,3,7,7,7,3,2,3,3,6,4,6,4,4,4,6,4,4,7,7,7,7,
	7,7,7,7,7,7,7,7,7,3,2,3,3,0,3,7,3,2,2,4,4,4,6,4,4,4,6,4,6,7,7,7,
	7,7,7,7,7,7,7,7,7,2,3,7,2,3,0,2,3,3,1,6,4,4,4,4,6,4,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,1,2,3,0,3,3,3,2,4,4,7,6,4,6,4,4,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,2,3,2,3,7,7,7,7,7,4,4,4,4,6,4,7,7,7,7,7,
	7,7,7,7,7,7,3,2,3,2,3,3,7,5,5,7,7,7,7,7,7,7,4,6,4,4,6,7,7,7,7,7,
	7,7,7,7,7,3,2,3,2,3,2,2,3,6,7,7,7,7,7,7,7,7,7,0,6,4,4,7,7,7,7,7,
	7,7,7,7,3,2,3,3,2,3,2,3,2,3,7,7,7,7,7,7,7,7,7,7,4,4,6,7,7,7,7,7,
	7,7,7,7,2,3,7,3,2,3,7,3,2,3,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,3,2,2,2,7,7,7,7,2,3,2,3,7,7,7,5,1,1,5,1,7,7,7,7,7,7,7,7,7,
	7,7,7,7,3,3,2,3,7,7,3,2,3,2,7,7,7,5,1,5,5,1,5,1,5,7,7,7,7,7,7,7,
	7,7,7,3,2,3,2,7,7,7,7,2,3,2,3,7,5,1,5,1,1,5,1,5,1,5,7,7,7,7,7,7,
	7,7,7,7,2,3,3,3,2,3,3,3,2,3,7,7,1,5,7,7,5,5,7,5,1,5,7,7,7,7,7,7,
	7,7,7,7,3,2,3,2,3,2,2,3,3,2,5,5,1,1,5,7,7,7,7,1,5,1,5,7,7,7,7,7,
	7,7,7,7,7,3,2,3,2,3,3,2,3,7,7,1,5,5,5,5,7,7,5,5,5,1,7,7,7,7,7,7,
	7,7,7,7,7,4,0,2,3,2,2,2,4,6,7,7,5,1,1,7,7,7,7,1,1,1,5,7,7,7,7,7,
	7,7,7,7,6,4,7,4,4,4,4,4,4,4,7,5,1,5,7,5,5,5,7,5,5,5,7,7,7,7,7,7,
	7,7,7,7,7,7,6,4,6,4,4,6,4,4,4,7,5,1,5,1,5,1,1,5,1,5,7,7,7,7,7,7,
	7,7,7,7,7,7,4,4,4,6,4,4,4,6,4,7,7,5,1,5,1,5,5,1,5,7,7,7,7,7,7,7,
	7,7,7,7,7,7,6,4,6,4,4,6,7,4,4,7,7,7,7,5,1,5,1,5,6,7,7,7,7,7,7,7,
	7,7,7,7,7,7,4,4,4,4,0,7,7,7,7,7,7,7,7,7,4,4,6,6,4,4,7,7,7,7,7,7,
	7,7,7,7,7,7,6,4,6,4,6,7,7,7,7,7,7,7,7,7,7,4,4,4,6,7,7,7,7,7,7,7,
	7,7,7,7,7,7,6,4,4,4,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,4,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
};

const unsigned short bmp3[32*32]={
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,6,6,2,7,7,6,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,6,6,7,3,6,6,7,7,2,6,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,6,6,0,6,6,6,6,0,6,6,7,7,6,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,6,7,0,6,6,6,2,6,2,6,6,6,2,4,6,2,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,6,6,6,2,6,6,6,6,6,0,6,6,2,6,6,6,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,3,6,6,2,6,2,0,2,0,2,0,2,6,6,2,6,1,6,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,5,2,6,6,2,0,6,0,6,0,6,0,6,2,6,6,6,6,7,7,7,7,7,7,7,
	7,7,7,7,7,7,6,6,6,2,6,0,6,0,2,1,2,0,6,0,6,6,2,6,3,7,7,7,7,7,7,7,
	7,7,7,7,7,7,2,6,6,6,0,3,0,6,0,6,2,5,2,0,2,6,6,0,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,6,2,6,2,4,2,2,0,6,0,2,0,6,0,2,6,6,6,0,7,7,7,7,7,7,
	7,7,7,7,7,7,4,2,6,6,0,0,6,0,7,0,2,0,6,2,0,6,6,6,6,7,7,7,7,7,7,7,
	7,7,7,7,7,2,6,6,6,2,0,6,0,2,0,2,4,2,0,5,2,6,2,2,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,2,4,6,2,0,7,0,6,2,0,7,2,0,6,6,6,6,2,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,2,6,6,2,0,2,2,4,2,0,4,2,2,2,6,6,6,6,7,7,7,7,7,7,
	7,7,7,7,7,7,6,6,6,6,6,2,6,0,4,3,4,2,2,6,6,6,6,1,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,2,5,2,6,6,0,6,2,0,2,4,6,2,6,6,6,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,6,6,2,6,6,6,6,6,2,6,6,2,6,2,6,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,6,7,6,2,6,2,6,6,2,6,7,7,5,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,2,4,7,1,6,6,1,6,6,0,7,6,0,5,2,0,0,1,6,7,7,7,7,7,
	7,7,7,7,7,7,4,0,4,0,0,4,2,4,7,7,2,7,5,0,4,0,4,4,6,4,0,7,7,7,7,7,
	7,7,7,7,7,7,7,0,1,4,0,4,0,0,6,7,0,0,0,4,0,4,0,0,0,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,4,0,4,0,5,4,0,5,6,4,0,0,0,5,0,5,4,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,1,4,0,6,0,6,0,1,7,4,1,6,0,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,2,7,0,7,7,0,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,4,3,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,0,4,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,0,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,0,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,4,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,0,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,0,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
};

const unsigned short bmp4[32*32]={
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,4,6,4,7,4,7,7,7,7,7,7,7,7,7,7,2,2,2,2,2,7,7,7,7,7,7,7,7,7,7,7,
	6,4,0,4,0,4,0,4,4,5,6,2,2,2,2,2,2,2,2,2,2,2,6,7,7,7,7,7,6,5,6,5,
	7,4,7,4,4,5,4,0,4,2,2,6,2,2,2,2,2,2,2,2,2,2,2,7,7,4,4,0,4,4,0,6,
	7,4,0,7,0,6,2,7,2,6,2,2,7,6,2,2,2,0,2,2,2,2,2,2,6,0,4,4,2,4,6,5,
	7,3,4,4,4,2,6,2,2,2,2,6,2,2,2,2,2,2,2,2,2,2,2,2,2,5,4,5,4,5,0,6,
	7,4,4,0,7,2,2,6,2,6,7,2,2,2,2,2,2,2,3,2,2,2,2,6,2,6,0,4,0,6,4,7,
	7,7,4,4,2,6,2,2,6,2,2,2,2,2,2,2,0,2,2,0,2,2,2,2,2,2,5,4,4,1,4,7,
	7,7,1,4,2,2,2,6,3,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,6,0,4,4,4,7,
	7,7,7,4,2,6,2,2,6,2,2,7,0,2,0,2,2,2,2,2,0,2,2,2,2,2,3,4,5,4,0,7,
	7,7,7,0,2,2,2,6,2,2,2,0,2,2,2,2,2,2,3,2,2,2,2,2,2,6,4,4,0,4,7,7,
	7,7,7,7,2,2,2,2,2,0,2,2,0,2,2,2,0,2,6,2,2,2,2,2,2,2,3,4,4,0,7,7,
	7,7,7,7,2,2,7,2,2,2,2,2,2,2,0,2,2,2,2,2,7,2,2,2,2,2,0,0,4,5,7,7,
	7,7,7,7,2,2,2,2,2,2,2,2,2,0,2,2,0,2,2,2,2,2,0,2,2,0,2,4,5,7,7,7,
	7,7,7,7,2,2,2,2,2,2,2,2,3,6,2,2,6,2,2,2,6,2,2,2,0,2,4,5,7,7,7,7,
	7,7,7,7,2,2,2,2,2,2,2,2,2,2,6,2,2,2,6,2,3,2,2,2,2,5,7,7,7,7,7,7,
	7,7,7,7,0,2,2,2,2,2,2,2,2,2,2,2,6,2,2,2,2,2,2,0,2,7,7,7,7,7,7,7,
	7,7,7,5,2,2,0,2,2,2,6,2,6,2,2,6,2,2,7,2,2,2,2,2,2,7,7,7,7,7,7,7,
	7,7,7,4,4,4,2,2,2,2,2,2,2,2,6,2,2,6,2,0,2,2,0,2,7,7,7,7,7,7,7,7,
	7,7,7,0,4,5,2,2,2,2,2,2,2,2,2,2,6,2,2,2,2,2,2,2,2,7,7,7,7,7,7,7,
	7,7,7,4,0,4,5,0,2,0,2,2,2,2,6,3,2,2,2,2,2,2,2,2,7,7,7,7,7,7,7,7,
	7,7,4,0,5,4,4,2,2,2,2,0,2,2,2,2,2,2,2,2,2,7,7,7,7,7,7,7,7,7,7,7,
	7,7,4,6,4,0,4,5,0,2,4,2,0,2,2,2,2,2,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,5,0,4,5,0,4,4,5,4,5,4,5,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,4,7,4,6,4,4,4,0,4,0,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,6,4,0,4,0,5,0,4,5,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,4,0,6,5,6,4,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,6,5,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
};

const unsigned short bmp5[32*32]={
	7,7,7,7,7,7,7,7,7,7,7,7,7,2,2,2,2,2,3,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,2,2,7,2,2,2,2,3,3,2,2,2,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,2,2,2,2,2,2,2,3,2,2,2,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,6,2,2,2,6,2,2,2,2,2,2,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,6,2,2,2,2,6,2,2,2,2,2,2,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,6,2,2,2,6,2,2,2,2,2,2,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,6,2,2,2,2,6,2,2,2,2,2,2,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,2,6,2,2,6,2,2,2,2,2,2,2,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,2,6,2,2,2,2,2,2,2,2,2,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,2,2,2,2,2,2,6,2,2,2,2,2,2,6,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,6,2,2,2,2,2,2,6,2,2,2,2,2,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,2,2,2,2,2,2,2,2,2,2,2,6,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,2,2,2,2,2,2,2,2,2,2,7,7,7,4,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,2,2,2,2,2,2,2,2,7,7,7,7,0,7,7,7,7,7,7,7,
	7,7,7,7,7,7,5,4,7,7,7,7,7,7,6,0,2,7,7,7,7,7,7,4,4,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,4,7,7,7,7,7,7,7,4,4,7,7,7,7,7,4,4,4,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,4,4,5,7,7,7,7,6,4,6,7,7,7,7,4,4,0,5,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,4,4,4,4,7,7,7,7,4,4,7,7,7,4,4,4,4,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,5,4,4,4,4,7,7,7,4,4,7,7,4,0,4,4,4,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,4,4,4,4,4,4,6,4,6,7,4,4,4,4,0,4,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,4,4,6,4,4,4,4,4,0,4,4,6,4,0,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,4,4,4,4,4,4,4,4,4,4,4,4,4,4,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,4,4,4,6,4,6,4,0,4,4,6,4,4,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,4,4,4,4,4,4,4,4,6,4,4,0,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,4,4,4,6,4,4,4,4,4,4,4,4,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,4,4,4,4,4,4,0,6,4,4,4,4,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,4,4,6,4,4,4,4,4,4,4,0,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,4,4,4,4,0,6,4,4,4,4,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,5,4,4,4,4,4,4,4,0,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,4,4,0,4,4,4,4,5,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,4,4,0,4,7,7,7,7,7,7,7,7,7,7,7,7,7,7
};

// 14x13ドットキャラクターデータ
const unsigned short bmp1413_1[14*13]={
	0,0,0,0,0,2,2,2,2,0,0,0,0,0,
	0,0,0,2,2,2,2,2,2,2,2,0,0,0,
	0,0,2,2,2,2,2,2,2,2,2,2,0,0,
	0,2,7,7,2,2,2,2,7,7,2,2,2,0,
	0,7,7,7,7,2,2,7,7,7,7,2,2,0,
	0,8,8,7,7,2,2,8,8,7,7,2,2,0,
	2,8,8,7,7,2,2,8,8,7,7,2,2,2,
	2,2,7,7,2,2,2,2,7,7,2,2,2,2,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	2,2,2,2,0,2,2,2,2,0,2,2,2,2,
	0,2,2,0,0,0,2,2,0,0,0,2,2,0
};
const unsigned short bmp1413_2[14*13]={
	0,0,0,0,0,5,5,5,5,0,0,0,0,0,
	0,0,0,5,5,5,5,5,5,5,5,0,0,0,
	0,0,5,5,5,5,5,5,5,5,5,5,0,0,
	0,5,7,7,5,5,5,5,7,7,5,5,5,0,
	0,7,7,7,7,5,5,7,7,7,7,5,5,0,
	0,8,8,7,7,5,5,8,8,7,7,5,5,0,
	5,8,8,7,7,5,5,8,8,7,7,5,5,5,
	5,5,7,7,5,5,5,5,7,7,5,5,5,5,
	5,5,5,5,5,5,5,5,5,5,5,5,5,5,
	5,5,5,5,5,5,5,5,5,5,5,5,5,5,
	5,5,5,5,5,5,5,5,5,5,5,5,5,5,
	5,5,0,5,5,5,0,0,5,5,5,0,5,5,
	5,0,0,0,5,5,0,0,5,5,0,0,0,5
};

void raymain(void); // レイトレーシングプログラム(raytrace.c)

void waitdraw(int n){
	//60分のn秒ウェイト
	drawcount=0;
	while(drawcount<n) ;
}
void testapp(void) {
	int i,j,k;
	unsigned short *ad1,*ad2,tempbuf[X_RES];
	int x1,y1,dx1,dy1,old_x1,old_y1;
	int x2,y2,dx2,dy2,old_x2,old_y2;
	int c=1, n = 32, br;

	while (1) {
		g_clearscreen();
		//カラーパレット設定
		for(i=0;i<8;i++){
			set_palette(i,255*(i&1),255*((i>>1)&1),255*(i>>2));
		}
		for(i=0;i<8;i++){
			set_palette(i+8,128*(i&1),128*((i>>1)&1),128*(i>>2));
		}
		//上部にアルファベット表示
		k=0;
		for(i=0;i<=24;i+=8){
			for(j=0;j<=X_RES-8;j+=8){
				g_putfont(j,i,k%16,0,k%26+'A');
				k++;
			}
		}

		//カラーパターン
		for(i=42;i<54;i++){
			for(j=0;j<X_RES;j++){
				g_pset(j,i,(X_RES-1-j)/(X_RES/8));
			}
		}
		for(;i<65;i++){
			for(j=0;j<X_RES;j++){
				g_pset(j,i,(X_RES-1-j)/(X_RES/8)+8);
			}
		}
		//32x32キャラクター
		g_putbmpmn(0,68,32,32,bmp1);
		g_putbmpmn(32,68,32,32,bmp2);
		g_putbmpmn(64,68,32,32,bmp3);
		g_putbmpmn(96,68,32,32,bmp4);
		g_putbmpmn(128,68,32,32,bmp5);
		g_putbmpmn(160,68,32,32,bmp1);
		g_putbmpmn(192,68,32,32,bmp2);
		g_putbmpmn(224,68,32,32,bmp3);
		g_putbmpmn(256,68,32,32,bmp4);
		g_putbmpmn(288,68,32,32,bmp5);

		//直線
		for(i=0;i<=120;i+=8){
			g_gline(i,102,0,Y_RES-1-i,4);
		}

		//円
		for(i=1;i<15;i++){
			g_circle(70,159,i*4,i);
		}

		//長方形
		g_gline(134,102,X_RES-1,102,7);
		g_gline(134,102,134,Y_RES-1,7);
		g_gline(X_RES-1,102,X_RES-1,Y_RES-1,7);
		g_gline(134,Y_RES-1,X_RES-1,Y_RES-1,7);

		//移動キャラクターの設定
		x1=135;
		y1=120;
		old_x1=x1;
		old_y1=y1;
		dx1=1;
		dy1=1;

		x2=140;
		y2=150;
		old_x2=x2;
		old_y2=y2;
		dx2=1;
		dy2=0;
		drawcount=0;
		n=0;
		k=0;
		while(n<=100){
			//映像区間終了待ち（約60分の1秒のウェイト）
			waitdraw(1);

			//アルファベット部分スクロール
			ad1=VRAM;
			ad2=tempbuf;
			for(i=0;i<X_RES;i++) *ad2++=*ad1++;
			ad2=VRAM;
			for(i=0;i<31;i++){
				for(j=0;j<X_RES;j++){
					*ad2++=*ad1++;
				}
			}
			ad1=tempbuf;
			for(i=0;i<X_RES;i++) *ad2++=*ad1++;

		//カウンタ表示
			if(++k==6){
				k=0;
				g_printnum(0,32,7,0,n++);
				if(n==0) g_printstr(0,32,7,0,"          ");
			}

		//キャラクターの移動
			g_clrbmpmn(old_x1,old_y1,14,13);	//移動前のキャラクターの消去
			g_clrbmpmn(old_x2,old_y2,14,13);	//移動前のキャラクターの消去

			g_putbmpmn(x1,y1,14,13,bmp1413_1);	//移動後のキャラクターの表示
			g_putbmpmn(x2,y2,14,13,bmp1413_2);	//移動後のキャラクターの表示

			old_x1=x1;
			old_y1=y1;
			x1+=dx1;
			y1+=dy1;
			if((x1+dx1<135) || (x1+dx1>X_RES-1-14)) dx1=-dx1;
			if((y1+dy1<103) || (y1+dy1>Y_RES-1-13)) dy1=-dy1;

			old_x2=x2;
			old_y2=y2;
			x2+=dx2;
			y2+=dy2;
			if((x2+dx2<135) || (x2+dx2>X_RES-1-14)) dx2=-dx2;
			if((y2+dy2<103) || (y2+dy2>Y_RES-1-13)) dy2=-dy2;
		}
		g_clearscreen();
		// フォントを画面いっぱい表示
		for (i = 0; i < 216; i += 8) {
			for (j = 0; j < X_RES; j += 8) {
				g_putfont(j, i, c, -1, n);
				n=(n-31)%96+32;
			}
			c=c%7+1;
		}
		waitdraw(3*60); //3秒ウェイト
		g_clearscreen();
		//カラーパレット設定
		for(i=0;i<256;i++) set_palette(i,(i&3)*85,((i&0x1c)>>2)*36,((i&0xe0)>>5)*36);
		//塗りつぶし円をランダムに表示
		for (i = 1; i <= 1000; i++) {
			g_circlefill(rand() % 320, rand() % 220, rand() % 50 + 10, rand() % 255+1);
			g_printnum(0,0,255,0,i);
		}
		waitdraw(3*60); //3秒ウェイト
		g_clearscreen();
		//塗りつぶし四角をランダムに表示
		for (i = 1; i <= 1000; i++) {
			g_boxfill(rand() % 320, rand() % 220, rand() % 320, rand() % 220, rand() % 255+1);
			g_printnum(0,0,255,0,i);
		}
		waitdraw(3*60); //3秒ウェイト
		g_clearscreen();
		//カラーパレット設定
		set_palette(0, 0, 0, 0);
		for (br = 1; br <= 146; br++) {
			for (i = 1; i <= 7; i++) {
				set_palette((br - 1)*7 + i, ((i&1)*br*7+1)/4, (((i&2)>>1)*br*7+1)/4, (((i&4)>>2)*br*7+1)/4);
			}
		}

//		stop_composite(); //映像出力停止で高速描画
		raymain(); // レイトレーシング実行
//		start_composite(); //映像出力再開

		waitdraw(20*60); //20秒ウェイト
	}
}
void main(void) {
	__builtin_mtc0(16, 0, (__builtin_mfc0(16, 0) & 0xfffffff8 | 3)); // Enable Cache

	CHECONbits.PFMWS = 3; // Set Program Flash Memory Wait State
//	ウェイト数を2にすることで高速化（定格外）

//	CHECONbits.PREFEN = 1; // Enable Instruction's Prefetch
//	チップRev.A1のエラッタ対策でプリフェッチは無効化

//	PRISS = 0x00100000;                /* assign shadow set #1 to priority level #5 ISRs */

	TRISB=0x10; // PORTBの出力設定（RB8はINPUT専用ピン）

// For debug configuration
//	PR3=0xffff;
//	T3CON=0x8000;

	init_composite(); //SPIコンポジット出力システムの初期化と表示開始

	testapp();
}
